<?php

namespace App\Http\Controllers;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use App\Models\Donation;
use App\Models\User;
use Razorpay\Api\Api;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Illuminate\Support\Facades\Mail;
use App\Mail\DonationInvoice;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;


use Barryvdh\DomPDF\Facade\Pdf;

class DonationController extends Controller
{


  public function downloadInvoiceByToken($token)
{
    // Find the donation by token
    $donation = Donation::where('token', $token)->firstOrFail();

    // Configure DomPDF options through the PDF facade
    $pdf = PDF::loadView('pdfs.donation_invoice', compact('donation'));

    // Set options like this:
    $pdf->setOption('isRemoteEnabled', true);
    $pdf->setOption('isHtml5ParserEnabled', true);
    $pdf->setOption('defaultFont', 'sans-serif'); // Optional: set default font

    // Set paper size and orientation
    $pdf->setPaper('A4', 'portrait');

    // Generate the file name
    $fileName = 'invoice_' . $donation->transaction_id . '.pdf';

    // Download the PDF
    return $pdf->download($fileName);
}

public function downloadInvoiceByTokenuser($token)
{
    try {
        $donation = Donation::where('token', $token)->firstOrFail();

        $pdf = PDF::loadView('pdfs.donation_invoice', compact('donation'))
            ->setOption('isRemoteEnabled', true)
            ->setOption('isHtml5ParserEnabled', true)
            ->setOption('defaultFont', 'sans-serif')
            ->setPaper('A4', 'portrait');

        $fileName = 'invoice_' . $donation->transaction_id . '.pdf';

        return $pdf->download($fileName);

    } catch (\Exception $e) {
        // Log the error
        Log::error('Invoice download failed: ' . $e->getMessage());

        // Redirect back with error message
        return back()->with('error', 'Failed to generate invoice. Please try again.');
    }
}
    public function showDonationForm()
    {
        return view('donation.form', [
            'user' => auth()->user()
        ]);
    }

   public function processDonation(Request $request)
{
    $validator = Validator::make($request->all(), [
        'amount' => 'required|numeric|min:100',
        'gateway' => 'required|in:razorpay,stripe',
        'name' => 'required|string|max:255',
        'email' => 'required|email|max:255',
        'phone' => 'required|string|max:20|regex:/^[0-9\+\-\(\)\s]+$/',
        'address' => 'required|string|max:500',
        'state' => 'required|string|max:100',
        'city' => 'required|string|max:100',
        'pincode' => 'required|digits:6',
    ], [
        'pincode.digits' => 'The PIN code must be exactly 6 digits.',
        'phone.regex' => 'Please enter a valid phone number.',
    ]);

    if ($validator->fails()) {
        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Please fix the validation errors'
            ], 422);
        }
        return redirect()->back()
            ->withErrors($validator)
            ->withInput();
    }

    $amount = $request->amount * 100; // Convert to paisa/cents

    try {
        if ($request->gateway == 'razorpay') {
            return $this->processRazorpayPayment($request, $amount);
        } else {
            return $this->processStripePayment($request, $amount);
        }
    } catch (\Exception $e) {
        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed: ' . $e->getMessage()
            ], 500);
        }
        return redirect()->back()
            ->with('error', 'Payment processing failed: ' . $e->getMessage())
            ->withInput();
    }
}

   protected function processRazorpayPayment($request, $amount)
{
    $api = new Api(env('RAZORPAY_KEY'), env('RAZORPAY_SECRET'));

    $order = $api->order->create([
        'amount' => $amount,
        'currency' => 'INR',
        'receipt' => 'donation_' . now()->timestamp,
        'payment_capture' => 1
    ]);

    // Store donation details in session
    session()->put('donation_details', [
        'amount' => $amount,
        'gateway' => 'razorpay',
        'user_data' => $request->only(['name', 'email', 'phone', 'address', 'state', 'city', 'pincode']),
        'create_account' => $request->has('create_account') && !auth()->check(),
        'razorpay_order_id' => $order->id
    ]);

    // Return JSON response with Razorpay options for direct payment
    return response()->json([
        'success' => true,
        'options' => [
            'key' => env('RAZORPAY_KEY'),
            'amount' => $amount,
            'currency' => 'INR',
            'name' => config('app.name'),
            'description' => 'Donation Payment',
            'image' => asset('images/logo.png'),
            'order_id' => $order->id,
            'handler' => route('razorpay.callback'),
            'prefill' => [
                'name' => $request->name,
                'email' => $request->email,
                'contact' => $request->phone
            ],
            'notes' => [
                'address' => $request->address,
                'state' => $request->state,
                'city' => $request->city,
                'pincode' => $request->pincode
            ],
            'theme' => [
                'color' => '#3399cc'
            ]
        ]
    ]);
}

  protected function processStripePayment($request, $amount)
{
    Stripe::setApiKey(env('STRIPE_SECRET'));

    // Store donation details in session
    session()->put('donation_details', [
        'amount' => $amount,
        'gateway' => 'stripe',
        'user_data' => $request->only(['name', 'email', 'phone', 'address', 'state', 'city', 'pincode']),
        'create_account' => $request->has('create_account') && !auth()->check()
    ]);

    $lineItems = [
        'price_data' => [
            'currency' => 'inr',
            'product_data' => [
                'name' => 'Donation',
                'description' => 'Charitable donation',
            ],
            'unit_amount' => $amount,
        ],
        'quantity' => 1,
    ];

    // For Indian export compliance, include address in metadata
    $metadata = [
        'donor_name' => $request->name,
        'donor_email' => $request->email,
        'donor_phone' => $request->phone,
        'donor_address' => $request->address,
        'donor_city' => $request->city,
        'donor_state' => $request->state,
        'donor_pincode' => $request->pincode,
        'donor_country' => 'IN',
        'export_compliant' => 'true'
    ];

    // Create the Checkout Session with proper Indian export compliance
    $session = Session::create([
        'payment_method_types' => ['card'],
        'line_items' => [$lineItems],
        'mode' => 'payment',
        'success_url' => route('donate.complete') . '?session_id={CHECKOUT_SESSION_ID}',
        'cancel_url' => route('donate'),
        'customer_email' => $request->email,
        'metadata' => $metadata,
        'billing_address_collection' => 'required', // This ensures address is collected
        'submit_type' => 'donate',
    ]);

    return redirect($session->url);
}
public function handleRazorpayCallback(Request $request)
{
    Log::info('Razorpay Callback Received:', $request->all());

    try {
        // Retrieve donation details from session
        $donationDetails = session('donation_details');

        if (!$donationDetails) {
            throw new \Exception('Donation session expired or not found');
        }

        $api = new Api(env('RAZORPAY_KEY'), env('RAZORPAY_SECRET'));

        $attributes = [
            'razorpay_order_id' => $request->razorpay_order_id,
            'razorpay_payment_id' => $request->razorpay_payment_id,
            'razorpay_signature' => $request->razorpay_signature
        ];

        $api->utility->verifyPaymentSignature($attributes);

        $donation = $this->createDonationRecord(
            $donationDetails['user_data'],
            $donationDetails['amount'] / 100,
            'razorpay',
            $request->razorpay_payment_id,
            $donationDetails['create_account']
        );

        // Clear session data
        session()->forget('donation_details');

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'donation' => $donation
            ]);
        }

        return redirect()->route('donate')->with([
            'success' => 'Thank you for your donation!'
        ]);

    } catch (\Exception $e) {
        Log::error('Razorpay callback error:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed: '.$e->getMessage()
            ], 400);
        }

        return redirect()->route('donate')->with('error', 'Payment processing failed: '.$e->getMessage());
    }
}

   public function completeDonation(Request $request)
{
    if ($request->has('session_id')) {
        // Stripe payment completion
        Stripe::setApiKey(env('STRIPE_SECRET'));
        $session = Session::retrieve($request->session_id);

        if ($session->payment_status == 'paid') {
            $donationDetails = session('donation_details');
            $amount = $donationDetails['amount'] / 100; // Convert back to rupees

            $donation = $this->createDonationRecord(
                $donationDetails['user_data'],
                $amount,
                'stripe',
                $session->payment_intent,
                $donationDetails['create_account']
            );

            session()->forget('donation_details');

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'donation' => $donation
                ]);
            }

            return redirect()->route('donate')->with([
                'success' => 'Thank you for your donation! Your invoice has been sent to your email address.'
            ]);
        }
    }

    if ($request->ajax()) {
        return response()->json([
            'success' => false,
            'message' => 'Payment verification failed.'
        ], 400);
    }

    return redirect()->route('donate')->with('error', 'Payment verification failed.');
}

protected function createDonationRecord($userData, $amount, $gateway, $transactionId, $createAccount = false)
{
    $user = null;

    // Check if user is authenticated
    if (auth()->check()) {
        $user = auth()->user();
        // Update user details if needed...
    } else {
        // Check if a user with the provided email already exists
        $user = User::where('email', $userData['email'])->first();

        if (!$user && $createAccount) {
            // Create a new user if none exists and account creation is requested
            $user = User::create([
                'name' => $userData['name'],
                'email' => $userData['email'],
                'password' => Hash::make($userData['email']),
                'phone' => $userData['phone'],
                'address' => $userData['address'],
                'state' => $userData['state'],
                'city' => $userData['city'],
                'pincode' => $userData['pincode'],
            ]);
        } elseif (!$user) {
            // Create a minimal guest user record if no user exists and no account creation
            $user = User::create([
                'name' => $userData['name'],
                'email' => $userData['email'],
                'password' => Hash::make($userData['email']),
                'role' => 'guest', // Ensure 'role' column exists in users table
            ]);
        }
    }

    // Create donation with guaranteed user_id

    $donation = new Donation();
        $donation->user_id = $user->id;
        $donation->amount = $amount;
        $donation->transaction_id = $transactionId;
        $donation->payment_gateway = $gateway;
        $donation->status = 'completed';
        $donation->name = $userData['name'];
        $donation->email = $userData['email'];
        $donation->phone = $userData['phone'];
        $donation->address = $userData['address'];
        $donation->state = $userData['state'];
        $donation->city = $userData['city'];
        $donation->pincode = $userData['pincode'];
        $donation->country = 'India';
         $donation->token = Str::random(32);

        $donation->save();


    Mail::to($donation->email)->send(new DonationInvoice($donation));

    return $donation;
}

   public function handleStripeWebhook(Request $request)
{
    $payload = $request->getContent();
    $sig_header = $request->header('Stripe-Signature');
    $endpoint_secret = env('STRIPE_WEBHOOK_SECRET');

    try {
        $event = \Stripe\Webhook::constructEvent(
            $payload, $sig_header, $endpoint_secret
        );
    } catch (\UnexpectedValueException $e) {
        // Invalid payload
        return response('Invalid payload', 400);
    } catch (\Stripe\Exception\SignatureVerificationException $e) {
        // Invalid signature
        return response('Invalid signature', 400);
    }

    // Handle the event
    switch ($event->type) {
        case 'checkout.session.completed':
            $session = $event->data->object;

            if ($session->payment_status == 'paid') {
                $this->handleSuccessfulStripePayment($session);
            }
            break;

        case 'payment_intent.succeeded':
            $paymentIntent = $event->data->object;
            // Additional handling if needed
            break;

        default:
            Log::info('Received unhandled Stripe event type: ' . $event->type);
    }

    return response('Success', 200);
}

protected function handleSuccessfulStripePayment($session)
{
    // Get donation details from session or metadata
    $donationDetails = session('donation_details');

    if (!$donationDetails) {
        // Fallback to metadata if session expired
        $donationDetails = [
            'user_data' => [
                'name' => $session->metadata->donor_name ?? '',
                'email' => $session->metadata->donor_email ?? '',
                'phone' => $session->metadata->donor_phone ?? '',
                'address' => $session->metadata->donor_address ?? '',
                'state' => $session->metadata->donor_state ?? '',
                'city' => $session->metadata->donor_city ?? '',
                'pincode' => $session->metadata->donor_pincode ?? ''
            ],
            'create_account' => false,
            'gateway' => 'stripe'
        ];
    }

    $amount = $session->amount_total / 100; // Convert back to rupees

    $this->createDonationRecord(
        $donationDetails['user_data'],
        $amount,
        'stripe',
        $session->payment_intent,
        $donationDetails['create_account']
    );
}

// In DonationController.php
public function showSuccess()
{
    if (!session('donation')) {
        return redirect()->route('donate');
    }

    return view('donation');
}


 public function index(Request $request)
{
    $status = $request->query('status', 'all');
    $user = auth()->user();

    $donations = Donation::where('user_id', $user->id)
        ->when($status !== 'all', function($query) use ($status) {
            return $query->where('status', $status);
        })
        ->orderBy('created_at', 'desc')
        ->paginate(10)
        ->appends(['status' => $status]); // Add this line
        // dd($donations);

    return view('donation.index', compact('donations', 'status'));
}
 public function show(Donation $donation)
    {

        return view('donation.show', compact('donation'));
    }

     protected function processPayment(Donation $donation)
    {
        // Simulate payment processing
        sleep(2);

        // For demo, 80% success, 15% pending, 5% failed
        $rand = rand(1, 100);

        if ($rand <= 80) {
            $status = 'success';
            $transaction_id = 'TXN' . strtoupper(Str::random(8));
        } elseif ($rand <= 95) {
            $status = 'pending';
            $transaction_id = null;
        } else {
            $status = 'failed';
            $transaction_id = null;
        }

        $donation->update([
            'status' => $status,
            'transaction_id' => $transaction_id,
        ]);
    }


   public function receipt(Donation $donation)
{


    // For PDF download
    if (request()->has('download')) {
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('donations.receipt', compact('donation'));
        return $pdf->download('receipt-'.$donation->transaction_id.'.pdf');
    }

    return view('donation.receipt', compact('donation'));
}



 public function ddindex()
    {
        $donations = Donation::orderBy('created_at', 'desc')->get();
        return view('admin.donations.index', compact('donations'));
    }

    public function ddshow(Donation $donation)
    {
        return view('admin.donations.show', compact('donation'));
    }

    public function dddownloadInvoice(Donation $donation)
    {
        $pdf = Pdf::loadView('admin.donations.invoice-pdf', compact('donation'));
        return $pdf->download('donation-invoice-'.$donation->transaction_id.'.pdf');
    }
}
